/******************************************************************************
* File:    tb_util.h
* 
* Purpose: Import testbench macros and functions.
* Author:  Quinten Brooks, Freescale Semiconductor
* Email:   quinten.brooks@freescale.com
* Date:    2007.03.21
******************************************************************************/

#ifndef __TB_UTIL_H
#define __TB_UTIL_H 1


/******************************************************************************
* The following macros can be used to access mapped memory via indirect
* addressing (pointer).
******************************************************************************/
#define MEM_PTR8(x)     (*(vuint8_t*)(x))       // 8-bit memory pointer
#define MEM_PTR16(x)    (*(vuint16_t*)(x))      // 16-bit memory pointer
#define MEM_PTR32(x)    (*(vuint32_t*)(x))      // 32-bit memory pointer


/******************************************************************************
* The following macros are used with the C API mailbox implementation to
* control miscellaneous testbench functions.
******************************************************************************/
#define MB_DUT_RST      0x00    // Reset DUT using external RESET pin
#define SET_GPIO        0x01    // Mailbox used to set pin drivers
#define GET_GPIO        0x02    // Mailbox used to set pin monitor
#define READ_GPIO       0x03    // Mailbox used to store monitored values
#define MB_DUMP_RAM     0x04    // Dump RAM contents to file
#define MB_DUMP_ROM     0x05    // Dump Flash contents to file
#define MB_DUMP_IFR     0x06    // Dump IFR Block contents to file
#define MB_CONNECT	0x07    // Connect GPIO pins
#define MB_RELEASE	0x08    // Release pin connections
#define MB_SW_EXT_CLK	0x09    // Switch to external clock
#define MB_GLOBAL_MASK  0x0A    // Control strobing in patterns for the tester

/******************************************************************************
 * The following are used to control strobing in tester targetted patterns
 ********************************************************************************/
#define COMPAREALL	0x00
#define MASKALL		0x01
#define STROBE		0x02

// Define pulse (wait 'x' ns, assert for 'y' ns)
#define DRV_PULSE(x,y)  (((uint32_t)(x) << 16) + (y))


/******************************************************************************
* The following macros are used with the C API mailbox implementation to
* control testbench GPIO pin drivers.
******************************************************************************/

// Define GPIO pin driver mailbox values
#define DRV0            0x000   // Drive 0
#define DRV1            0x100   // Drive 1
#define DRVZ            0x200   // Drive Z
#define DRV_OFF         0x300   // Disable pin driver
#define CHIP2_PIN_GPIOA0	79 
#define CHIP2_PIN_GPIOA1	78 
#define CHIP2_PIN_GPIOA2 	77 
#define CHIP2_PIN_GPIOA3 	76 
#define CHIP2_PIN_GPIOA4 	75 
#define CHIP2_PIN_GPIOA5 	74 
#define CHIP2_PIN_GPIOA6 	73 
#define CHIP2_PIN_GPIOA7	72
#define CHIP2_PIN_GPIOB0 	71
#define CHIP2_PIN_GPIOB1 	70
#define CHIP2_PIN_GPIOB2 	69
#define CHIP2_PIN_GPIOB3 	68
#define CHIP2_PIN_GPIOB4 	67
#define CHIP2_PIN_GPIOB5 	66
#define CHIP2_PIN_GPIOB6 	65 
#define CHIP2_PIN_GPIOB7 	64 
#define CHIP2_PIN_GPIOC0 	63 
#define CHIP2_PIN_GPIOC1 	62 
#define CHIP2_PIN_GPIOC2 	61 
#define CHIP2_PIN_GPIOC3 	60 
#define CHIP2_PIN_GPIOC4 	59 
#define CHIP2_PIN_GPIOC5 	58
#define CHIP2_PIN_GPIOC6 	57
#define CHIP2_PIN_GPIOC7 	56
#define CHIP2_PIN_GPIOD0 	55
#define CHIP2_PIN_GPIOD1 	54
#define CHIP2_PIN_GPIOD2 	53
#define CHIP2_PIN_GPIOD3 	52
#define CHIP2_PIN_GPIOE0 	51
#define CHIP2_PIN_GPIOE1 	50 
#define CHIP2_PIN_GPIOE2 	49 
#define CHIP2_PIN_GPIOE3 	48 
#define CHIP2_PIN_GPIOE4 	47 
#define CHIP2_PIN_GPIOE5 	46 
#define CHIP2_PIN_GPIOE6 	45 
#define CHIP2_PIN_GPIOE7 	44 
#define CHIP2_PIN_GPIOF0 	43
#define CHIP2_PIN_GPIOF1 	42
#define CHIP2_PIN_GPIOF2 	41
#define CHIP2_PIN_GPIOF3 	40
#define PIN_GPIOA0			39 
#define PIN_GPIOA1			38 
#define PIN_GPIOA2 			37 
#define PIN_GPIOA3 			36 
#define PIN_GPIOA4 			35 
#define PIN_GPIOA5 			34 
#define PIN_GPIOA6 			33 
#define PIN_GPIOA7			32
#define PIN_GPIOB0 			31
#define PIN_GPIOB1 			30
#define PIN_GPIOB2 			29
#define PIN_GPIOB3 			28
#define PIN_GPIOB4 			27
#define PIN_GPIOB5 			26
#define PIN_GPIOB6 			25 
#define PIN_GPIOB7 			24 
#define PIN_GPIOC0 			23 
#define PIN_GPIOC1 			22 
#define PIN_GPIOC2 			21 
#define PIN_GPIOC3 			20 
#define PIN_GPIOC4 			19 
#define PIN_GPIOC5 			18
#define PIN_GPIOC6 			17
#define PIN_GPIOC7 			16
#define PIN_GPIOD0 			15
#define PIN_GPIOD1 			14
#define PIN_GPIOD2 			13
#define PIN_GPIOD3 			12
#define PIN_GPIOE0 			11
#define PIN_GPIOE1 			10 
#define PIN_GPIOE2 			9 
#define PIN_GPIOE3 			8 
#define PIN_GPIOE4 			7 
#define PIN_GPIOE5 			6 
#define PIN_GPIOE6 			5 
#define PIN_GPIOE7 			4 
#define PIN_GPIOF0 			3
#define PIN_GPIOF1 			2
#define PIN_GPIOF2 			1
#define PIN_GPIOF3 			0



/******************************************************************************
* The following macros are used to perform flash operations via software.
******************************************************************************/
// Flash Command Byte Values
#define FCMD_VERIFY     0x05    // Erase Verify
#define FCMD_PROG       0x20    // Byte Program
#define FCMD_BPROG      0x25    // Burst Program
#define FCMD_S_ERASE    0x40    // Sector Erase
#define FCMD_M_ERASE    0x41    // Mass Erase

// Flash Clock Divider Values
#define FCDIV_FCLK_HI   0x00
#define FCDIV_FCLK_LO   0x00
#define FCDIV_FCLK(x)   ((x < 10) ? FCDIV_FCLK_LO: FCDIV_FCLK_HI)
// HFM Values
#define CMD_VERIFY      0x05	// HFM Command for Erase Verify
#define CMD_PROGRAM     0x20	// HFM Command for Program
#define CMD_PAGE_ERASE  0x40	// HFM Command for Page Erase
#define CMD_MASS_ERASE  0x41	// HFM Command for Mass Erase

#define CMD_RDARYM	0x06	// HFM Command for Read Array into MISR
#define CMD_MRDARYM0	0x1C	// HFM Command for Margin Read Array Zeroes into MISR
#define CMD_MRDARYM1	0x1D	// HFM Command for Margin Read Array Ones into MISR
#define CMD_RDARYMI	0x66	// HFM Command for Read IFR into MISR
#define CMD_MRDARYMI0	0x76	// HFM Command for Margin Read IFR Zeroes into MISR
#define CMD_MRDARYMI1	0x77	// HFM Command for Margin Read IFR Ones into MISR
#define HFMCLKD_DIV     0x1     // HFM Clock Value (0x14 for previous chips)

// Flash Info
#define PFLASH_PAGE_SIZE        256     // PFLASH Page Size (in words)
#define PFLASH_BLOCK_SIZE       0x2000 // PFLASH Block Size (in words)
#define HFM_CUTOFF_FREQ         12800000  //
#define HFM_IN_RANGE_FREQ       199000   //

#define HFM_CLKD				((OSC_FREQ/HFM_IN_RANGE_FREQ)-1)
#define HFM_CLKD_EXT_CLK		(hfmPRDIV8 | (((EXT_CLK_FREQ/8)/HFM_IN_RANGE_FREQ)-1))
#define HFM_CLKD_FAST_SIM		0x0 // For quick HFM prog/erase, not compatible with flash block timing checks

// Functions
#define pflash_mass_erase               flash_control(CMD_MASS_ERASE,(uint16_t*)&_pflash_start,0,1)
#define pflash_page_erase(x)            flash_control(CMD_PAGE_ERASE,(uint16_t*)x,0,1)
#define pflash_write_array(x,y,z)       flash_control(CMD_PROGRAM,(uint16_t*)x,(uint16_t*)y,z)
#define pflash_write(x,y)               {flash_data=y; flash_control(CMD_PROGRAM,(uint16_t*)x,(uint16_t*)&flash_data,1);}
#define pflash_misr(x,y)		flash_control(CMD_RDARYM,(uint16_t*)x,(uint16_t*)y,1);
#define pflash_misr_margin_ones(x,y)	flash_control(CMD_MRDARYM1,(uint16_t*)x,(uint16_t*)y,1);
#define pflash_misr_margin_zeroes(x,y)	flash_control(CMD_MRDARYM0,(uint16_t*)x,(uint16_t*)y,1);
#define ifr_misr(x,y)			flash_control(CMD_RDARYMI,(uint16_t*)x,(uint16_t*)y,1);
#define ifr_misr_margin_ones(x,y)	flash_control(CMD_MRDARYMI1,(uint16_t*)x,(uint16_t*)y,1);
#define ifr_misr_margin_zeroes(x,y)	flash_control(CMD_MRDARYMI0,(uint16_t*)x,(uint16_t*)y,1);


/******************************************************************************
* The following variable is used to assign values to the flash configuration
* field.  This controls the flash security, protection, and back door access
* features of the flash module.
******************************************************************************/
extern const uint16_t flash_config[];


#endif __TB_UTIL_H

/* End of "tb_util.h" */
